/*
	File:		FinderWindow.cp

	Contains:	Finder window simulating using the Appearance Manager.

	Version:	Appearance 1.0 SDK

	Copyright:	 1997 by Apple Computer, Inc., all rights reserved.

	File Ownership:

		DRI:				Edward Voas

		Other Contact:		7 of 9, Borg Collective

		Technology:			OS Technologies Group

	Writers:

		(edv)	Ed Voas

	Change History (most recent first):

		 <1>	 9/11/97	edv		First checked in.
*/

//
//	This module demonstrates an Appearance adoption example for a Finder-like window.
//	It uses actual Appearance Manager primitives to draw some of the window elements,
//	such as the window header and placard. Normally, it would be best to use the
//	window header and placard controls, since they take care of enabling and disabling
//	themselves. This, however, is a good example of how to use lower level routines
//	and still be theme savvy.
//
//	Although this window has scroll bars, it doesn't actually scroll anything. It's
//	more an example of getting the visual appearance together.
//

#include <MacWindows.h>
#include "FinderWindow.h"
#include "Appearance.h"
#include "ColorPenState.h"
#include "Assertions.h"

#define width( r )		( (r).right - (r).left )
#define height( r )		( (r).bottom - (r).top )

FinderWindow::FinderWindow() : BaseWindow( 128 )
{				
	Rect		rect;
	
		// Note the use of the new defProc constants here.
		// This eliminates the need to go thru the mapping
		// CDEF for scroll bars, which would normally happen
		// after calling RegisterAppearanceClient.
		
	CalcVertScrollBarRect( rect );
	fVertScrollBar = NewControl( fWindow, &rect, "\p", true, 0, 0, 100, kControlScrollBarProc, 0 );
	
	CalcHorizScrollBarRect( rect );
	fHorizScrollBar = NewControl( fWindow, &rect, "\p", true, 0, 0, 100, kControlScrollBarProc, 0 );	
}

FinderWindow::~FinderWindow()
{
	if ( fVertScrollBar )
		DisposeControl( fVertScrollBar );
	
	if ( fHorizScrollBar )
		DisposeControl( fHorizScrollBar );
}

//
//	 Activate
//
//	Activates the contents of the window.
//
void
FinderWindow::Activate( EventRecord& )
{
	Rect		rect = fWindow->portRect;
	
	SetPort( fWindow );
	InsetRect( &rect, -1, -1 );
	rect.bottom = rect.top + 40;
	::DrawThemeWindowHeader( &rect, kThemeStateActive );
	ValidRect( &rect );
	
	DrawPlacard( kThemeStateActive, true );

		// Here we use the new ActivateControl call. We
		// could have still used HiliteControl, but this
		// is much more straightforward. It also works
		// right with and without embedding, so it's a
		// real good idea to start using it.
		
	ActivateControl( fHorizScrollBar );
	ActivateControl( fVertScrollBar );
}

//
//	 Deactivate
//
//	Deactivates the contents of the window.
//
void
FinderWindow::Deactivate( EventRecord& )
{
	Rect		rect = fWindow->portRect;
	
	SetPort( fWindow );
	InsetRect( &rect, -1, -1 );
	rect.bottom = rect.top + 40;
	::DrawThemeWindowHeader( &rect, kThemeStateDisabled );
	ValidRect( &rect );
	
	DrawPlacard( kThemeStateDisabled, true );
	
		// Here we use the new DeactivateControl call. We
		// could have still used HiliteControl, but this
		// is much more straightforward. It also works
		// right with and without embedding, so it's a
		// real good idea to start using it.
		
	DeactivateControl( fHorizScrollBar );
	DeactivateControl( fVertScrollBar );
}

//
//	 Draw
//
//	Draws our window. Appearance Manager calls handle multiple devices properly. We
//	need to call DeviceLoop to handle drawing our list view, though.
//
void
FinderWindow::Draw()
{
	Rect					rect = fWindow->portRect;
	ThemeDrawState			drawState;
	RgnHandle				rgn;
	DeviceLoopDrawingUPP	proc;
	
	UpdateControls( fWindow, fWindow->visRgn );
	
	drawState = ((WindowPeek)fWindow)->hilited ? kThemeStateActive : kThemeStateDisabled;
	
	::SetPort( fWindow );
	::InsetRect( &rect, -1, -1 );
	rect.bottom = rect.top + 40;
	::DrawThemeWindowHeader( &rect, drawState );
	
	GetContentRect( rect );

	rgn = NewRgn();
	RectRgn( rgn, &rect );
	
	proc = NewDeviceLoopDrawingProc( DrawListView );
	DeviceLoop( rgn, proc, (long)this, 0 );
	DisposeRoutineDescriptor( proc );
	
	DisposeRgn( rgn );

	DrawPlacard( drawState, false );
}

//
//	 DrawListView
//
//	Draws our list view columns, using the correct theme brushes.
//
pascal void
FinderWindow::DrawListView( SInt16 depth, SInt16 flags, GDHandle /*device*/, long userData )
{
	FinderWindow*	window = (FinderWindow*)userData;
	Rect			rect;
	ColorPenState	state;
	SInt16			baseLine;
	
	window->GetContentRect( rect );
	
		// Because the brushes used by the Appearance Manager could
		// be colors or patterns, our GetColorAndPenState routine
		// always saves the current background pattern as well as the
		// fore and back colors.
		
	::GetColorAndPenState( &state );

	::SetThemeBackground( kThemeListViewBackgroundBrush, depth, (flags & gdDevType) != 0 );
	::EraseRect( &rect );
	
	rect.left += 40;
	rect.right = rect.left + 120;
	
	::SetThemeBackground( kThemeListViewSortColumnBackgroundBrush, depth, (flags & gdDevType) != 0 );
	::EraseRect( &rect );

	::SetThemePen( kThemeListViewSeparatorBrush, depth, (flags & gdDevType) != 0 );

	window->GetContentRect( rect );
	
	for ( baseLine = rect.top; baseLine <= rect.bottom; baseLine += 15 )
	{
		::MoveTo( rect.left, baseLine );
		::LineTo( rect.right - 1, baseLine );
	}
	
	::SetColorAndPenState( &state );
}

//
//	 Resize
//
//	Resize our window to the appropriate size specified in width and height. Make sure
//	the scroll bars are repositioned properly.
//
void
FinderWindow::Resize( SInt16 width, SInt16 height )
{
	Rect		horizRect, vertRect;
	
	if ( width > width( fWindow->portRect ) )
	{
		Rect	temp = fWindow->portRect;
		
		temp.left = temp.right - 5;
		temp.bottom = 40;
		InvalRect( &temp );
	}
	InvalidateScrollBars();
	InvalidatePlacard();
	
	BaseWindow::Resize( width, height );
	
	InvalidateScrollBars();
	InvalidatePlacard();
	
	CalcHorizScrollBarRect( horizRect );
	CalcVertScrollBarRect( vertRect );
	
	MoveControl( fHorizScrollBar, horizRect.left, horizRect.top );
	MoveControl( fVertScrollBar, vertRect.left, vertRect.top );
	SizeControl( fHorizScrollBar, width( horizRect ), height( horizRect ) );
	SizeControl( fVertScrollBar, width( vertRect ), height( vertRect ) );
}

//
//	 HandleClick
//
//	Simple routine to handle scroll bar tracking, even though they don't do anything.
//
void
FinderWindow::HandleClick( EventRecord& event )
{
	ControlHandle		control;
	SInt16				part;
	Point				localPt;
	ControlActionUPP	actionProc;
	
	SetPort( fWindow );
	localPt = event.where;
	GlobalToLocal( &localPt );
	
	part = FindControl( localPt, fWindow, &control );
	switch ( part )
	{
		case kControlUpButtonPart:
		case kControlDownButtonPart:
		case kControlPageUpPart:
		case kControlPageDownPart:
			actionProc = NewControlActionProc( ScrollBarAction );
			TrackControl( control, localPt, actionProc );
			DisposeRoutineDescriptor( actionProc );
			break;
		
		case kControlIndicatorPart:
			TrackControl( control, localPt, (ControlActionUPP)-1L );
			break;
	}
}

//
//	 GetContentRect
//
//	Get our content rect, which is the area not including the scroll bars and the
//	window header. It is basically the list view area.
//
void
FinderWindow::GetContentRect( Rect& rect )
{
	rect = fWindow->portRect;
	
	rect.bottom -= 15;
	rect.right -= 15;
	rect.top += 39;
}

//
//	 DrawPlacard
//
//	Draws our placard next to the horizontal scroll bar.
//
void
FinderWindow::DrawPlacard( ThemeDrawState state, Boolean validate )
{
	Rect		rect;
	
	rect = fWindow->portRect;
	
	rect.bottom++;
	rect.left--;
	rect.top = rect.bottom - 16;
	rect.right = 121;
	
	::DrawThemePlacard( &rect, state );
	if ( validate )
		ValidRect( &rect );
}

//
//	 InvalidateScrollBars
//
//	Invalidates the scroll bar areas.
//
void
FinderWindow::InvalidateScrollBars()
{
	Rect		tempRect;
	
	SetPort( fWindow );
	CalcHorizScrollBarRect( tempRect );
	InvalRect( &tempRect );
	EraseRect( &tempRect );
	
	CalcVertScrollBarRect( tempRect );
	InvalRect( &tempRect );
	EraseRect( &tempRect );
}

//
//	 InvalidatePlacard
//
//	Invalidates the placard area.
//
void
FinderWindow::InvalidatePlacard()
{
	Rect		rect;
	
	rect = fWindow->portRect;
	
	rect.bottom++;
	rect.top = rect.bottom - 16;
	rect.right = 121;
	rect.left--;

	InvalRect( &rect );
	EraseRect( &rect );
}

//
//	 CalcHorizScrollBarRect
//
//	Calculates the position where the horizontal scroll bar should be placed.
//
void
FinderWindow::CalcHorizScrollBarRect( Rect& rect )
{
	rect = fWindow->portRect;
	rect.bottom++;
	rect.left = rect.left + 120;
	rect.top = rect.bottom - 16;
	rect.right -= 14;
}

//
//	 CalcVertScrollBarRect
//
//	Calculates the position where the vertical scroll bar should be placed.
//
void
FinderWindow::CalcVertScrollBarRect( Rect& rect )
{
	rect = fWindow->portRect;
	rect.right++;
	rect.left = rect.right - 16;
	rect.bottom -= 14;
	rect.top = 38;
}

//
//	 ScrollBarAction
//
//	A simple callback to give some feedback when clicking the scroll arrows or page
//	up/down areas.
//
pascal void
FinderWindow::ScrollBarAction( ControlHandle control, SInt16 part )
{
	switch ( part )
	{
		case kControlUpButtonPart:
			if ( GetControlValue( control) > GetControlMinimum( control ) )
				SetControlValue( control, GetControlValue( control ) - 1 );
			break;
		
		case kControlDownButtonPart:
			if ( GetControlValue( control) < GetControlMaximum( control ) )
				SetControlValue( control, GetControlValue( control ) + 1 );
			break;
			
		case kControlPageUpPart:
			if ( GetControlValue( control) > GetControlMinimum( control ) )
				SetControlValue( control, GetControlValue( control ) - 10 );
			break;
			
		case kControlPageDownPart:
			if ( GetControlValue( control) < GetControlMaximum( control ) )
				SetControlValue( control, GetControlValue( control ) + 10 );
			break;
	}			
}

